from typing import Callable, Optional, List, Union
from ..models.models import (
    ExtendedClient,
    ExtendedUser,
    ExtendedGroup,
    GroupMember,
    GroupStatistics,
    CharmStats,
    Achievements,
    UserAchievements,
)
from ..types.types import Language, DeviceType, OnlineState


class Information:
    def __init__(self, client, client_profile: ExtendedClient):
        self.client = client
        self.client_profile = client_profile

    @property
    def ClientProfile(self):
        return self.client_profile

    def request_user(
        self,
        user_id: int,
        callback: Optional[Callable[[ExtendedUser], None]] = None
    ):
        def on_success(data):
            user = ExtendedUser()
            if isinstance(data, dict):
                for key, value in data.items():
                    if hasattr(user, key):
                        setattr(user, key, value)
            if callback:
                callback(user)

        self.client.write_packet(
            self.client.packet.user_profile(user_id, True),
            adv=True,
            success=on_success
        )

    def request_group(
        self,
        search: Union[str, int],
        callback: Optional[Callable[[ExtendedGroup], None]] = None
    ):
        def on_success(data):
            group = ExtendedGroup()
            if isinstance(data, dict):
                for key, value in data.items():
                    if hasattr(group, key):
                        setattr(group, key, value)
            if callback:
                callback(group)

        self.client.write_packet(
            self.client.packet.group_profile(search, True),
            adv=True,
            success=on_success
        )

    def request_groups(
        self,
        callback: Optional[Callable[[List[ExtendedGroup]], None]] = None
    ):
        def on_success(data):
            groups = []
            if isinstance(data, list):
                for item in data:
                    group = ExtendedGroup()
                    if isinstance(item, dict):
                        for key, value in item.items():
                            if hasattr(group, key):
                                setattr(group, key, value)
                    groups.append(group)
            if callback:
                callback(groups)

        self.client.write_packet(
            self.client.packet.group_list(),
            adv=True,
            success=on_success
        )

    def group_member_list(
        self,
        group_id: int,
        callback: Optional[Callable[[List[GroupMember]], None]] = None
    ):
        def on_success(data):
            members = []
            if isinstance(data, list):
                for item in data:
                    member = GroupMember()
                    if isinstance(item, dict):
                        for key, value in item.items():
                            if hasattr(member, key):
                                setattr(member, key, value)
                    members.append(member)
            if callback:
                callback(members)

        self.client.write_packet(
            self.client.packet.group_member_list(group_id),
            adv=True,
            success=on_success
        )

    def request_group_stats(
        self,
        group_id: int,
        callback: Optional[Callable[[GroupStatistics], None]] = None
    ):
        def on_success(data):
            stats = GroupStatistics()
            if isinstance(data, dict):
                for key, value in data.items():
                    if hasattr(stats, key):
                        setattr(stats, key, value)
            if callback:
                callback(stats)

        self.client.write_packet(
            self.client.packet.request_group_statistics(group_id),
            adv=True,
            success=on_success
        )

    def request_user_active_charms(
        self,
        user_id: int,
        callback: Optional[Callable[[List], None]] = None
    ):
        self.client.write_packet(
            self.client.packet.request_user_active_charms(user_id),
            adv=True,
            success=callback
        )

    def request_user_expired_charms(
        self,
        user_id: int,
        callback: Optional[Callable[[List], None]] = None
    ):
        self.client.write_packet(
            self.client.packet.request_user_expired_charms(user_id),
            adv=True,
            success=callback
        )

    def request_user_charm_statistics(
        self,
        user_id: int,
        callback: Optional[Callable[[CharmStats], None]] = None
    ):
        def on_success(data):
            stats = CharmStats()
            if isinstance(data, dict):
                for key, value in data.items():
                    if hasattr(stats, key):
                        setattr(stats, key, value)
            if callback:
                callback(stats)

        self.client.write_packet(
            self.client.packet.request_user_charm_stats(user_id),
            adv=True,
            success=on_success
        )

    def request_user_achievements(
        self,
        user_id: int,
        callback: Optional[Callable[[List[UserAchievements]], None]] = None
    ):
        self.client.write_packet(
            self.client.packet.request_user_achievements(user_id),
            adv=True,
            success=callback
        )

    def request_achievement_list(
        self,
        language: Language = Language.ENGLISH,
        callback: Optional[Callable[[List[Achievements]], None]] = None
    ):
        self.client.write_packet(
            self.client.packet.request_achievements(language),
            adv=True,
            success=callback
        )

    def update_online_state(self, state: OnlineState):
        self.client.update_state(state)

    def update_user_profile(self, profile: ExtendedUser):
        self.client.write_packet(
            self.client.packet.update_user_profile(profile),
            adv=True
        )

    def update_name(self, name: str):
        if self.client_profile and self.client_profile.extended:
            self.client_profile.extended['name'] = name
            self.update_user_profile(self.client_profile)

    def update_charm(self, charm_id: int):
        from ..models.models import SelectedList
        charm = SelectedList()
        charm.charm_id = charm_id
        charm.position = 0

    def user_add(self, user_id: int, message: str = ''):
        self.client.write_packet(
            self.client.packet.user_add(user_id, message),
            adv=True
        )

    def user_delete(self, user_id: int):
        self.client.write_packet(
            self.client.packet.user_delete(user_id),
            adv=True
        )

    def user_block(self, user_id: int):
        self.client.write_packet(
            self.client.packet.user_block(user_id),
            adv=True
        )

    def notifications(
        self,
        language: Language = Language.ENGLISH,
        device_type: DeviceType = DeviceType.WEB,
        callback: Optional[Callable] = None
    ):
        self.client.write_packet(
            self.client.packet.notifications(language, device_type),
            adv=True,
            success=callback
        )

    def clear_notification(self):
        self.client.write_packet(
            self.client.packet.clear_notifications(),
            adv=True
        )

    def request_topics(
        self,
        language: Language = Language.ENGLISH,
        callback: Optional[Callable] = None
    ):
        self.client.write_packet(
            self.client.packet.request_topics(language),
            adv=True,
            success=callback
        )

    def request_discover_groups(
        self,
        language: Language = Language.ENGLISH,
        max_results: int = 25,
        callback: Optional[Callable] = None
    ):
        self.client.write_packet(
            self.client.packet.request_discover_groups(language, max_results),
            adv=True,
            success=callback
        )

    def conversation_list(self, callback: Optional[Callable] = None):
        self.client.write_packet(
            self.client.packet.conversation_list(),
            adv=True,
            success=callback
        )

    def message_history(
        self,
        id_: int,
        timestamp: str,
        is_group: bool = True,
        before: bool = True,
        callback: Optional[Callable] = None
    ):
        self.client.write_packet(
            self.client.packet.message_history(id_, timestamp, is_group, before),
            adv=True,
            success=callback
        )

    def create_group(self, group_data):
        self.client.write_packet(
            self.client.packet.create_group(group_data),
            adv=True
        )

    def update_group(self, group_data):
        self.client.write_packet(
            self.client.packet.update_group_profile(group_data),
            adv=True
        )
