import os

import time
import logging
from kazoo.client import KazooClient
from datetime import datetime


def look_for_missed_broker_id(broker_ids, min_broker_id, max_broker_id):
    # loop from 1001 to 1006 included
    for broker_id in range(min_broker_id, max_broker_id + 1, 1):
        if str(broker_id) not in broker_ids:
            return broker_id

    return -1


def create_or_update_meta(log_dir, meta_file, broker_id):
    # loop from 1001 to 1006 included
    if not os.path.exists(log_dir):
        try:
            os.makedirs(log_dir)
        except OSError as exception:
            print("Create log dir (%s) error, the error is %s" % (log_dir, exception.message))
            # stop the following operation until error solved
            return

    # check whether it's brand new broker, if not, then return
    for topic_dir in os.listdir(log_dir):
        # if any topic exist in the kafka log directory, we will see it old kafka broker, then return directly
        if os.path.isdir(topic_dir):
            print("topic is exist in %s, so return directly." % topic_dir)
            return

    # check meta.properties exist
    meta_file_path = log_dir + "/" + meta_file;

    if os.path.exists(meta_file_path):
        # the new broker is created conflict, so reset the broker id to new option, then wait for kafka restart to solve
        # it again
        os.remove(meta_file_path)

    # create new meta.properties for the current broker
    try:
        meta_file = open(meta_file_path, 'w+')
        meta_file.write('#\n')
        meta_file.write('#' + str(datetime.now()) + ', generated by external script.\n')
        meta_file.write('version=0\n')
        meta_file.write('broker.id=' + str(broker_id) + '\n')
    except OSError as exception:
        print("Write file (%s) error, the error is %s." % (meta_file, exception.message))
        return


def manage_meta(zookeeper_url, log_dir, meta_file, min_broker_id, max_broker_id):
    # configure log
    logging.basicConfig()

    # Print the pass through parameters
    print (
        "Create kafka meta.properties if it is not exist with arg [zookeeper-url : %s, log-dir : %s, broker range in "
        "%d, %d]." % (zookeeper_url, log_dir, min_broker_id, max_broker_id))

    # Connect to zookeeper
    zk = KazooClient(hosts=zookeeper_url)
    zk.start()

    missed_broker_id = -1

    # retrieve brokers
    broker_path = "/brokers/ids"
    if zk.exists(broker_path):
        broker_ids = zk.get_children(broker_path)

    str_broker_ids = []
    if len(broker_ids) > 0:
        for broker_id in broker_ids:
            str_broker_ids.append(str(broker_id))

    missed_broker_id = look_for_missed_broker_id(str_broker_ids, min_broker_id, max_broker_id)

    if missed_broker_id != -1:
        # sleep seconds are aimed to avoid potential broker id conflict, especially for brand new cluster deployed
        sleep_seconds = (missed_broker_id - min_broker_id + 1) * 10
        print("Next broker id: %s, sleep: %d seconds." % (missed_broker_id, sleep_seconds))
        time.sleep(sleep_seconds)

        # create or update meta.properties
        create_or_update_meta(log_dir, meta_file, missed_broker_id)

    # disconnect to zookeeper
    zk.stop()


# if __name__ == "__main__":
#     """
#     This script is aime to control broker id generated strategy.
#
#     scenarios
#     ---------
#     1, broker id may be conflict between each other when try to create the whole kafka cluster with quasar on aws.
#     2, retirement one of the existed broker, the auto launched node with not expected broker id, e.g 1007, 1008.
#     3, especially for case 2), if the new node will different broker id, we need to do manual partition assignment.
#
#     what does this script do?
#     ---------
#     1, grab broker ids from remote zookeeper.
#     2, calculate the first excepted id from a indicated scope, e.g [1001, 1006].
#     3, create kafka log folder and meta.properties with the calculated id if it's a brand new node.
#     4, or 3), update the broker id existed in the meta.properties with the new id if there is no any topic in the log
#        folder, this case is aimed to solve the potential broker conflict when try to launch new cluster.
#     """
#
#     parser = argparse.ArgumentParser(description='Kafka meta.properties creation if it is not exist')
#     parser.add_argument('--zookeeper-url', default="localhost:2181", help='URL of zookeeper')
#     parser.add_argument('--log-dir', default="/kafka/data", help='Dir of kafka log')
#     parser.add_argument('--meta-file', default="meta.properties", help='Meta file of kafka')
#     parser.add_argument('--min-broker-id', type=int, default=1001, help='The min broker id, e.g 1001')
#     parser.add_argument('--max-broker-id', type=int, default=9999, help='The max broker id, e.g 9999')
#     args = parser.parse_args()
#
#
#
#     manage_meta(args.zookeeper_url, args.log_dir, args.meta_file, args.min_broker_id, args.max_broker_id)
