"""
Copyright (c) 2023, Zscaler Inc.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
"""

from zscaler.request_executor import RequestExecutor
from zscaler.utils import format_url, transform_common_id_fields, reformat_params
from zscaler.api_client import APIClient
from zscaler.zia.models.casb_malware_rules import CasbMalwareRules


class CasbMalwareRulesAPI(APIClient):

    _zia_base_endpoint = "/zia/api/v1"

    def __init__(self, request_executor):
        super().__init__()
        self._request_executor: RequestExecutor = request_executor

    def list_rules(
        self,
        query_params=None,
    ) -> tuple:
        """
        Returns a list of all Casb Malware Rules for the specified rule type.

        Args:
            query_params {dict}: Map of query parameters for the request.

                ``[query_params.search]`` {str}: Search string for filtering results.

                ``[query_params.rule_type]`` {str}: The type of rules to retrieve (e.g., "OFLCASB_AVP_ITSM").

                Supported Values: `ANY`, `NONE`, `OFLCASB_AVP_FILE`, `OFLCASB_AVP_EMAIL`, `OFLCASB_AVP_CRM`, 
                    `OFLCASB_AVP_ITSM`, `OFLCASB_AVP_COLLAB`, `OFLCASB_AVP_REPO`, `OFLCASB_AVP_STORAGE`,
                    `OFLCASB_AVP_GENAI`

        Returns:
            tuple: The list of Casb Malware Rules.

        Examples:
            List all rules for a specific type::

                >>> rules_list, _, error = client.zia.casb_malware_rules.list_rules(
                ... query_params={'rule_type': 'OFLCASB_AVP_REPO'})
                >>> if error:
                ...     print(f"Error listing casb malware rules rules: {error}")
                ...     return
                ... print(f"Total rules found: {len(rules_list)}")
                ... for rule in rules_list:
                ...     print(rule.as_dict())
        """
        http_method = "get".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /casbMalwareRules
        """
        )

        query_params = query_params or {}

        body = {}
        headers = {}

        request, error = self._request_executor.create_request(http_method, api_url, body, headers, params=query_params)

        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)

        if error:
            return (None, response, error)

        try:
            result = []
            for item in response.get_results():
                result.append(CasbMalwareRules(self.form_response_body(item)))
        except Exception as error:
            return (None, response, error)

        return (result, response, None)

    def get_rule(
        self,
        rule_id: int,
        rule_type: str,
    ) -> tuple:
        """
        Returns information for the specified casb malware rule under the specified rule type.

        Args:
            rule_id (str): The unique identifier for the casb malware rule.
            rule_type (str): The type of the rule (e.g., "OFLCASB_AVP_ITSM").

                Supported Values: `ANY`, `NONE`, `OFLCASB_AVP_FILE`, `OFLCASB_AVP_EMAIL`, `OFLCASB_AVP_CRM`, 
                    `OFLCASB_AVP_ITSM`, `OFLCASB_AVP_COLLAB`, `OFLCASB_AVP_REPO`, `OFLCASB_AVP_STORAGE`,
                    `OFLCASB_AVP_GENAI`

        Returns:
            :obj:`Tuple`: The resource record for the casb malware rule.

        Examples:
            Get a specific rule by ID and type::

                >>> fetched_rule, _, error = client.zia.casb_malware_rules.get_rule(
                ...     rule_type='OFLCASB_AVP_REPO',
                ...     rule_id='1072401',
                ... )
                >>> if error:
                ...     print(f"Error fetching rule by ID: {error}")
                ...     return
                ... print(f"Fetched rule by ID: {fetched_rule.as_dict()}")
        """
        http_method = "get".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /casbMalwareRules/{rule_id}
        """
        )

        params = {"ruleType": rule_type}

        body = {}
        headers = {}

        request, error = self._request_executor.create_request(http_method, api_url, body, headers, params=params)

        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request, CasbMalwareRules)

        if error:
            return (None, response, error)

        try:
            result = CasbMalwareRules(self.form_response_body(response.get_body()))
        except Exception as error:
            return (None, response, error)
        return (result, response, None)

    def list_all_rules(
        self,
    ) -> tuple:
        """
        Returns a list of all Casb Malware Rules.

        Args:
            N/A

        Returns:
            tuple: The list of all Casb Malware Rules.

        Examples:
            List all casb malware rules:

                >>> rules_list, _, error = client.zia.casb_malware_rules.list_all_rules(
                >>> if error:
                ...     print(f"Error listing all Casb Malware Rules rules: {error}")
                ...     return
                ... print(f"Total rules found: {len(rules_list)}")
                ... for rule in rules_list:
                ...     print(rule.as_dict())
        """
        http_method = "get".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /casbMalwareRules/all
        """
        )

        body = {}
        headers = {}

        request, error = self._request_executor.create_request(http_method, api_url, body, headers)

        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)

        if error:
            return (None, response, error)

        try:
            result = []
            for item in response.get_results():
                result.append(CasbMalwareRules(self.form_response_body(item)))
        except Exception as error:
            return (None, response, error)

        return (result, response, None)
    
    def add_rule(self, **kwargs) -> tuple:
        """
        Adds a new cloud app control rule.

        Args:
            name (str): Name of the rule.

        Keyword Args:
            order (str): The order of the rule, defaults to adding rule to bottom of list
            enabled (bool): The rule state

            type (str): The type of the rule (e.g., "OFLCASB_AVP_ITSM").

                Supported Values: `ANY`, `NONE`, `OFLCASB_AVP_FILE`, `OFLCASB_AVP_EMAIL`, `OFLCASB_AVP_CRM`, 
                    `OFLCASB_AVP_ITSM`, `OFLCASB_AVP_COLLAB`, `OFLCASB_AVP_REPO`, `OFLCASB_AVP_STORAGE`,
                    `OFLCASB_AVP_GENAI`

            cloud_app_tenant_ids (list):The list of cloud application tenants IDs for which the rule is applied
            bucket_ids (list): The list of buckets IDs for the Zscaler service to inspect for sensitive data
            labels (list): The list of label IDs that this rule applies to
            casb_email_label (dict): Name-ID of the email label associated with the rule
            casb_tombstone_template (dict): Name-ID of the quarantine tombstone template associated with the rule

        Returns:
            :obj:`Tuple`: New casb malware rule resource.

        Examples:
            casb malware rule for ITSM Access::

                >>> added_rule, _, error = client.zia.casb_malware_rules.add_rule(
                ...     name='GitLab_Tenant01',
                ...     type = "OFLCASB_AVP_REPO",
                ...     action = "OFLCASB_AVP_REPORT_MALWARE",
                ...     enabled=True,
                ...     order=1,
                ...     cloud_app_tenant_ids = [15881081],
                ...     labels = [1441065],
                ...     bucket_ids = [1442271, 1442270, 1442268, 1442269, 1442272],
                ... )
                >>> if error:
                ...     print(f"Error adding rule: {error}")
                ...     return
                ... print(f"Rule added successfully: {added_rule.as_dict()}")
                ... )
        """
        http_method = "post".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /casbMalwareRules
        """
        )

        body = kwargs

        if "enabled" in body:
            body["state"] = "ENABLED" if body.pop("enabled") else "DISABLED"

        transform_common_id_fields(reformat_params, body, body)

        request, error = self._request_executor.create_request(
            method=http_method,
            endpoint=api_url,
            body=body,
        )

        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request, CasbMalwareRules)
        if error:
            return (None, response, error)

        try:
            result = CasbMalwareRules(self.form_response_body(response.get_body()))
        except Exception as error:
            return (None, response, error)

        return (result, response, None)

    def update_rule(self, rule_id: str, **kwargs) -> tuple:
        """
        Updates an existing casb malware rule.

        Args:
            name (str): Name of the rule.

        Keyword Args:
            order (str): The order of the rule, defaults to adding rule to bottom of list
            enabled (bool): The rule state

            type (str): The type of the rule (e.g., "OFLCASB_AVP_ITSM").

                Supported Values: `ANY`, `NONE`, `OFLCASB_AVP_FILE`, `OFLCASB_AVP_EMAIL`, `OFLCASB_AVP_CRM`, 
                    `OFLCASB_AVP_ITSM`, `OFLCASB_AVP_COLLAB`, `OFLCASB_AVP_REPO`, `OFLCASB_AVP_STORAGE`,
                    `OFLCASB_AVP_GENAI`

            cloud_app_tenant_ids (list): The list of cloud application tenants IDs for which the rule is applied
            bucket_ids (list): The list of buckets IDs for the Zscaler service to inspect for sensitive data
            labels (list): The list of label IDs that this rule applies to
            casb_email_label (dict): Name-ID of the email label associated with the rule
            casb_tombstone_template (dict): Name-ID of the quarantine tombstone template associated with the rule

        Returns:
            :obj:`Tuple`: Existing Casb Malware Rules resource.

        Examples:
            Update an existing casb malware rule for ITSM Access::

                >>> updated_rule, _, error = client.zia.casb_malware_rules.update_rule(
                ...     rule_id='1072324',
                ...     name='GitLab_Tenant01',
                ...     type = "OFLCASB_AVP_REPO",
                ...     action = "OFLCASB_AVP_REPORT_MALWARE",
                ...     enabled=True,
                ...     order=1,
                ...     cloud_app_tenant_ids = [15881081],
                ...     labels = [1441065],
                ...     bucket_ids = [1442271, 1442270, 1442268, 1442269, 1442272],
                ... )
                >>> if error:
                ...     print(f"Error updating rule: {error}")
                ...     return
                ... print(f"Rule updated successfully: {updated_rule.as_dict()}")
                ... )
        """
        http_method = "put".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /casbMalwareRules/{rule_id}
        """
        )

        body = kwargs

        if "enabled" in body:
            body["state"] = "ENABLED" if body.pop("enabled") else "DISABLED"

        transform_common_id_fields(reformat_params, body, body)

        request, error = self._request_executor.create_request(
            method=http_method,
            endpoint=api_url,
            body=body,
        )

        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)

        if error:
            return (None, response, error)

        try:
            result = CasbMalwareRules(self.form_response_body(response.get_body()))
        except Exception as error:
            return (None, response, error)
        return (result, response, None)

    def delete_rule(self, rule_type: str, rule_id: int) -> tuple:
        """
        Deletes the specified Casb Malware Rules.

        Args:
            rule_id (int): The unique identifier for the Casb Malware Rules.

            rule_type (str): The type of the rule (e.g., "OFLCASB_AVP_ITSM").

                Supported Values: `ANY`, `NONE`, `OFLCASB_AVP_FILE`, `OFLCASB_AVP_EMAIL`, `OFLCASB_AVP_CRM`, 
                    `OFLCASB_AVP_ITSM`, `OFLCASB_AVP_COLLAB`, `OFLCASB_AVP_REPO`, `OFLCASB_AVP_STORAGE`,
                    `OFLCASB_AVP_GENAI`

        Returns:
            :obj:`int`: The status code for the operation.

        Examples:
            >>> _, _, error = client.zia.casb_malware_rules.delete_rule(
            ...     rule_type='OFLCASB_AVP_REPO',
            ...     rule_id='1072324'
            ... )
            >>> if error:
            ...     print(f"Error deleting rule: {error}")
            ...     return
            ... print(f"Rule with ID 1072324 deleted successfully.")
        """
        http_method = "delete".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /casbMalwareRules/{rule_id}
        """
        )
        params = {"ruleType": rule_type}

        request, error = self._request_executor.create_request(http_method, api_url, params=params)
        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)
        if error:
            return (None, response, error)

        return (None, response, None)
