"""
Copyright (c) 2023, Zscaler Inc.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
"""

from zscaler.oneapi_object import ZscalerObject
from zscaler.oneapi_collection import ZscalerCollection


class Microtenant(ZscalerObject):
    """
    Initialize the Microtenant model based on API response.

    Args:
        config (dict): A dictionary representing the microtenant configuration.
    """

    def __init__(self, config=None):
        super().__init__(config)
        if config:
            self.id = config["id"] if "id" in config else None
            self.modified_time = config["modifiedTime"] if config and "modifiedTime" in config else None
            self.creation_time = config["creationTime"] if config and "creationTime" in config else None
            self.modified_by = config["modifiedBy"] if config and "modifiedBy" in config else None
            self.name = config["name"] if config and "name" in config else None
            self.description = config["description"] if config and "description" in config else None
            self.enabled = config["enabled"] if config and "enabled" in config else None
            self.operator = config["operator"] if config and "operator" in config else None
            self.criteria_attribute = config["criteriaAttribute"] if config and "criteriaAttribute" in config else None

            self.privileged_approvals_enabled = (
                config["privilegedApprovalsEnabled"] if config and "privilegedApprovalsEnabled" in config else None
            )

            self.criteria_attribute_values = ZscalerCollection.form_list(
                config["criteriaAttributeValues"] if "criteriaAttributeValues" in config else [], str
            )

        else:
            self.id = None
            self.name = None
            self.description = None
            self.enabled = None
            self.creation_time = None
            self.modified_by = None
            self.modified_time = None
            self.operator = None
            self.criteria_attribute = None
            self.criteria_attribute_values = None
            self.privileged_approvals_enabled = None

    def request_format(self):
        """
        Formats the Segment Group data into a dictionary suitable for API requests.
        """
        parent_req_format = super().request_format()
        current_obj_format = {
            "id": self.id,
            "name": self.name,
            "description": self.description,
            "enabled": self.enabled,
            "modifiedTime": self.modified_time,
            "modifiedBy": self.modified_by,
            "creationTime": self.creation_time,
            "operator": self.operator,
            "criteriaAttribute": self.criteria_attribute,
            "criteriaAttributeValues": self.criteria_attribute_values,
            "privilegedApprovalsEnabled": self.privileged_approvals_enabled,
        }
        parent_req_format.update(current_obj_format)
        return parent_req_format
