"""
Copyright (c) 2023, Zscaler Inc.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
"""

from typing import Dict, List, Optional, Any, Union
from zscaler.api_client import APIClient
from zscaler.request_executor import RequestExecutor
from zscaler.zia.models.malware_protection_settings import MalwareSettings
from zscaler.utils import format_url
from zscaler.types import APIResult


class MalwareProtectionPolicyAPI(APIClient):
    """
    A Client object for the Malware Protection Policy resource.
    """

    _zia_base_endpoint = "/zia/api/v1"

    def __init__(self, request_executor: "RequestExecutor") -> None:
        super().__init__()
        self._request_executor: RequestExecutor = request_executor

    def get_atp_malware_policy(self) -> APIResult[dict]:
        """
        Retrieves the Malware Protection Policy configuration.

        Returns:
            tuple: A tuple containing:
                - dict: The current atp malware protection policy inspection with keys:
                    - block_unscannable_files (bool): Whether to block unscannable files.
                    - block_password_protected_archive_files (bool): Whether to block password-protected archive files.
                - Response: The raw HTTP response from the API.
                - error: An error message if the request failed, otherwise `None`.

        Examples:
            Returns malware policy settings:

            >>> settings, _, err = client.zia.malware_protection_policy.get_atp_malware_policy()
            ... if err:
            ...     print(f"Error fetching malware policy: {err}")
            ...     return
            ... print("Current malware policy fetched successfully.")
            ... print(settings)
        """
        http_method = "get".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /cyberThreatProtection/malwarePolicy
        """
        )

        request, error = self._request_executor.create_request(http_method, api_url)
        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)
        if error:
            return (None, response, error)

        try:
            policy_data = response.get_body()
            if not isinstance(policy_data, dict):
                raise ValueError("Unexpected response format: policy data should be a dictionary.")

            malware_policy = {
                "block_unscannable_files": policy_data.get("blockUnscannableFiles", False),
                "block_password_protected_archive_files": policy_data.get("blockPasswordProtectedArchiveFiles", False),
            }
            return (malware_policy, response, None)
        except Exception as ex:
            return (None, response, ex)

    def update_atp_malware_policy(
        self,
        block_unscannable_files: bool,
        block_password_protected_archive_files: bool
    ) -> APIResult[dict]:
        """
        Updates the Malware Protection Policy configuration.

        Args:
            block_unscannable_files (bool): Whether to block unscannable files.
            block_password_protected_archive_files (bool): Whether to block password-protected archive files.

        Returns:
            tuple:
                - Response: The raw HTTP response from the API.
                - error: An error message if the request failed, otherwise `None`.

        Examples:
            Configure Malware Policy using settings:

            >>> policy, _, err = client.zia.malware_protection_policy.update_atp_malware_policy(
            ...     block_unscannable_files=True,
            ...     block_password_protected_archive_files=True
            ... )
            >>> if err:
            ...     print(f"Error fetching malware policy: {err}")
            ...     return
            ... print("Current malware policy fetched successfully.")
            ... print(policy)
        """
        if not isinstance(block_unscannable_files, bool) or not isinstance(block_password_protected_archive_files, bool):
            raise TypeError("Both arguments must be of type bool.")

        http_method = "put".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /cyberThreatProtection/malwarePolicy
        """
        )

        payload = {
            "blockUnscannableFiles": block_unscannable_files,
            "blockPasswordProtectedArchiveFiles": block_password_protected_archive_files,
        }

        request, error = self._request_executor.create_request(http_method, api_url, payload)
        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)
        if error:
            return (None, response, error)

        try:
            updated_policy_data = response.get_body()
            if not isinstance(updated_policy_data, dict):
                raise ValueError("Unexpected response format: policy data should be a dictionary.")

            # Convert camelCase to snake_case
            updated_policy = {
                "block_unscannable_files": updated_policy_data.get("blockUnscannableFiles", False),
                "block_password_protected_archive_files": updated_policy_data.get("blockPasswordProtectedArchiveFiles", False),
            }
            return (updated_policy, response, None)
        except Exception as ex:
            return (None, response, ex)

    def get_atp_malware_inspection(self) -> APIResult[dict]:
        """
        Retrieves the traffic inspection configurations of Malware Protection policy

        Returns:
            tuple: A tuple containing:
                - dict: The current atp malware protection policy inspection with keys:
                    - inspect_inbound (bool): Enables or disables scanning of incoming internet traffic for malicious content
                    - inspect_outbound (bool): Enables or disables scanning of outgoing internet traffic for malicious content
                - Response: The raw HTTP response from the API.
                - error: An error message if the request failed, otherwise `None`.

        Examples:
            Returns malware inspection settings:

            >>> protocols, _, err = client.zia.malware_protection_policy.get_atp_malware_inspection()
            ... if err:
            ...     print(f"Error fetching malware inspection: {err}")
            ...     return
            ... print("Current malware inspection fetched successfully.")
            ... print(protocols)
        """
        http_method = "get".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /cyberThreatProtection/atpMalwareInspection
        """
        )

        request, error = self._request_executor.create_request(http_method, api_url)
        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)
        if error:
            return (None, response, error)

        try:
            policy_data = response.get_body()
            if not isinstance(policy_data, dict):
                raise ValueError("Unexpected response format: policy data should be a dictionary.")

            # Convert camelCase to snake_case
            malware_policy = {
                "inspect_inbound": policy_data.get("inspectInbound", False),
                "inspect_outbound": policy_data.get("inspectOutbound", False),
            }
            return (malware_policy, response, None)
        except Exception as ex:
            return (None, response, ex)

    def update_atp_malware_inspection(self, inspect_inbound: bool, inspect_outbound: bool) -> APIResult[dict]:
        """
        Updates the traffic inspection configurations of Malware Protection policy.

        Args:
            - inspect_inbound (bool): Enables or disables scanning of incoming internet traffic for malicious content
            - inspect_outbound (bool): Enables or disables scanning of outgoing internet traffic for malicious content

        Returns:
            tuple: A tuple containing:
                - dict: The updated atp malware protection inspection policy.
                - Response: The raw HTTP response from the API.
                - error: An error message if the request failed, otherwise `None`.

        Examples:
            Configure Malware Inspection using settings:

            >>> inspection, _, err = client.zia.malware_protection_policy.update_atp_malware_inspection(
            ...     inspect_inbound=True,
            ...     inspect_outbound=True
            ... )
            >>> if err:
            ...     print(f"Error fetching malware inspection: {err}")
            ...     return
            ... print("Current malware inspection fetched successfully.")
            ... print(inspection)
        """
        if not isinstance(inspect_inbound, bool) or not isinstance(inspect_outbound, bool):
            raise TypeError("Both arguments must be of type bool.")

        http_method = "put".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /cyberThreatProtection/atpMalwareInspection
        """
        )

        payload = {
            "inspectInbound": inspect_inbound,
            "inspectOutbound": inspect_outbound,
        }

        request, error = self._request_executor.create_request(http_method, api_url, payload)
        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)
        if error:
            return (None, response, error)

        try:
            updated_policy_data = response.get_body()
            if not isinstance(updated_policy_data, dict):
                raise ValueError("Unexpected response format: policy data should be a dictionary.")

            # Convert camelCase to snake_case
            updated_policy = {
                "inspect_inbound": updated_policy_data.get("inspectInbound", False),
                "inspect_outbound": updated_policy_data.get("inspectOutbound", False),
            }
            return (updated_policy, response, None)
        except Exception as ex:
            return (None, response, ex)

    def get_atp_malware_protocols(self) -> APIResult[dict]:
        """
        Retrieves the traffic protocols configurations of Malware Protection policy

        Returns:
            tuple: A tuple containing:
                - dict: The current atp malware protection policy protocols with keys:
                    - inspect_http (bool): Enables or disables scanning of HTTP traffic
                        (and HTTPS traffic if SSL Inspection is enabled) for malicious content in real time
                    - inspect_ftp_over_http (bool): Enables or or disables scanning of FTP over HTTP traffic
                        for malicious content in real time
                    - inspect_ftp (bool): Enables or disables scanning of FTP traffic for malicious content in real time

                - Response: The raw HTTP response from the API.
                - error: An error message if the request failed, otherwise `None`.

        Examples:
            Returns malware inspection settings:

            >>> protocols, _, err = client.zia.malware_protection_policy.get_atp_malware_protocols()
            ... if err:
            ...     print(f"Error fetching malware protocols: {err}")
            ...     return
            ... print("Current malware protocols fetched successfully.")
            ... print(protocols)
        """
        http_method = "get".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /cyberThreatProtection/atpMalwareProtocols
        """
        )

        request, error = self._request_executor.create_request(http_method, api_url)
        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)
        if error:
            return (None, response, error)

        try:
            policy_data = response.get_body()
            if not isinstance(policy_data, dict):
                raise ValueError("Unexpected response format: policy data should be a dictionary.")

            # Convert camelCase to snake_case
            malware_policy = {
                "inspect_http": policy_data.get("inspectHttp", False),
                "inspect_ftp_over_http": policy_data.get("inspectFtpOverHttp", False),
                "inspect_ftp": policy_data.get("inspectFtp", False),
            }
            return (malware_policy, response, None)
        except Exception as ex:
            return (None, response, ex)

    def update_atp_malware_protocols(
        self,
        inspect_http: bool,
        inspect_ftp_over_http: bool,
        inspect_ftp: bool
    ) -> APIResult[dict]:
        """
        Updates the traffic protocols configurations of Malware Protection policy.

        Args:
            - inspect_http (bool): Enables or disables scanning of HTTP traffic and HTTPS traffic if SSL Inspection is enabled
                for malicious content in real time
            - inspect_ftp_over_http (bool): Enables or disables scanning of FTP over HTTP traffic in real time
            - inspect_ftp (bool): Enables or disables scanning of FTP traffic for malicious content in real time

        Returns:
            tuple: A tuple containing:
                - dict: The updated atp malware protection protocols policy.
                - Response: The raw HTTP response from the API.
                - error: An error message if the request failed, otherwise `None`.

        Examples:
            Configure Malware Inspection using settings:

            >>> protocol, _, err = client.zia.malware_protection_policy.update_atp_malware_protocols(
            ...     inspect_http=True,
            ...     inspect_ftp_over_http=True
            ...     inspect_ftp=True
            ... )
            >>> if err:
            ...     print(f"Error fetching malware protocols: {err}")
            ...     return
            ... print("Current malware protocol fetched successfully.")
            ... print(protocol)
        """
        if (
            not isinstance(inspect_http, bool)
            or not isinstance(inspect_ftp_over_http, bool)
            or not isinstance(inspect_ftp, bool)
        ):
            raise TypeError("Both arguments must be of type bool.")

        http_method = "put".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /cyberThreatProtection/atpMalwareProtocols
        """
        )

        payload = {
            "inspectHttp": inspect_http,
            "inspectFtpOverHttp": inspect_ftp_over_http,
            "inspectFtp": inspect_ftp,
        }

        request, error = self._request_executor.create_request(http_method, api_url, payload)
        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)
        if error:
            return (None, response, error)

        try:
            updated_policy_data = response.get_body()
            if not isinstance(updated_policy_data, dict):
                raise ValueError("Unexpected response format: policy data should be a dictionary.")

            # Convert camelCase to snake_case
            updated_policy = {
                "inspect_http": updated_policy_data.get("inspectInbound", False),
                "inspect_ftp_over_http": updated_policy_data.get("inspectOutbound", False),
                "inspect_ftp": updated_policy_data.get("inspectFtp", False),
            }
            return (updated_policy, response, None)
        except Exception as ex:
            return (None, response, ex)

    def get_malware_settings(self) -> APIResult[dict]:
        """
        Retrieves the malware protection policy configuration details

        Returns:
            tuple: A tuple containing:
                - MalwareSettings: The current malware protection policy settings object.
                - Response: The raw HTTP response returned by the API.
                - error: An error message if the request failed; otherwise, `None`.

        Examples:
            Returns malware settings:

            >>> settings, _, err = client.zia.malware_protection_policy.get_malware_settings()
            ... if err:
            ...     print(f"Error fetching malware settings: {err}")
            ...     return
            ... print("Current malware settings fetched successfully.")
            ... print(settings)
        """
        http_method = "get".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /cyberThreatProtection/malwareSettings
        """
        )

        request, error = self._request_executor.create_request(http_method, api_url)

        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)

        if error:
            return (None, response, error)

        try:
            advanced_settings = MalwareSettings(response.get_body())
            return (advanced_settings, response, None)
        except Exception as ex:
            return (None, response, ex)

    def update_malware_settings(self, settings: MalwareSettings) -> APIResult[dict]:
        """
        Updates the malware protection policy configuration details

        Args:
            settings (:obj:`MalwareSettings`):
                An instance of `MalwareSettings` containing the updated configuration.

                Supported attributes:
                    **Malware Protection Policy Settings:**
                    - virus_blocked (bool): Allow/block malicious programs that can harm systems and data
                    - virus_capture (bool): Enable/disable packet capture (PCAP) for viruses
                    - unwanted_applications_blocked (bool): Allow/block unwanted files downloaded with user programs
                    - unwanted_applications_capture (bool): Enable/disable PCAP for unwanted applications
                    - trojan_blocked (bool): Allow/block Trojan viruses disguised as useful software
                    - trojan_capture (bool): Enable/disable PCAP for Trojan viruses
                    - worm_blocked (bool): Allow/block worms that replicate and spread malicious code
                    - worm_capture (bool): Enable/disable PCAP for worms
                    - adware_blocked (bool): Allow/block files that auto-display ads or install adware
                    - adware_capture (bool): Enable/disable PCAP for adware
                    - spyware_blocked (bool): Allow/block files that covertly collect user/org data
                    - spyware_capture (bool): Enable/disable PCAP for spyware
                    - ransomware_blocked (bool): Allow/block ransomware that encrypts files until ransom is paid
                    - ransomware_capture (bool): Enable/disable PCAP for ransomware
                    - remote_access_tool_blocked (bool): Allow/block downloads from known remote access tools
                    - remote_access_tool_capture (bool): Enable/disable PCAP for remote access tools

        Returns:
            tuple: A tuple containing:
                - MalwareSettings: The updated malware protection settings object.
                - Response: The raw HTTP response returned by the API.
                - error: An error message if the update failed; otherwise, `None`.

        Examples:
            Update Malware Settings by enabling Office365 and adjusting the session timeout:

            >>> settings, response, err = client.zia.malware_protection_policy.update_malware_settings()
            >>> if not err:
            ...     settings.virus_blocked = True
            ...     updated_settings, response, err = client.zia.malware_protection_policy.update_malware_settings(settings)
            ...     if not err:
            ...         print(f"Updated Virus Blocked: {updated_settings.virus_blocked}")
            ...     else:
            ...         print(f"Failed to update settings: {err}")
        """
        http_method = "put".upper()
        api_url = format_url(
            f"""
            {self._zia_base_endpoint}
            /cyberThreatProtection/malwareSettings
            """
        )

        payload = settings.request_format()

        request, error = self._request_executor.create_request(http_method, api_url, payload)

        if error:
            return (None, None, error)

        response, error = self._request_executor.execute(request)

        if error:
            return (None, response, error)

        return self.get_malware_settings()
